<?php
session_start(); // Iniciar a sessão

// Definir o número máximo de tentativas de login
define('MAX_LOGIN_ATTEMPTS', 5);
define('LOCKOUT_TIME', 30 * 60); // 30 minutos de bloqueio após tentativas excessivas

$error = ''; // Para armazenar mensagens de erro de login

// Conectar ao banco de dados SQLite
try {
    $db = new PDO("sqlite:database.db");
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Função para verificar tentativas de login
    function checkLoginAttempts($ipAddress, $db) {
        $stmt = $db->prepare("SELECT attempts, last_attempt FROM login_attempts WHERE ip_address = ?");
        $stmt->execute([$ipAddress]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($result) {
            $attempts = $result['attempts'];
            $lastAttempt = strtotime($result['last_attempt']);

            // Se o número de tentativas exceder o limite e o tempo de bloqueio não tiver passado
            if ($attempts >= MAX_LOGIN_ATTEMPTS && (time() - $lastAttempt) < LOCKOUT_TIME) {
                return false; // Bloquear login
            }
        }

        return true; // Permitir login
    }

    // Função para registrar ou atualizar as tentativas de login
    function logLoginAttempt($ipAddress, $success = false, $db) {
        try {
            $db->beginTransaction();

            if (!$success) {
                // Se a tentativa de login falhou, incrementar o número de tentativas
                $stmt = $db->prepare("SELECT attempts FROM login_attempts WHERE ip_address = ?");
                $stmt->execute([$ipAddress]);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);

                // Se já houver tentativas, incrementa, senão começa com 1
                $attempts = $result ? $result['attempts'] + 1 : 1;

                // Inserir ou substituir os dados (usando INSERT OR REPLACE para SQLite)
                $stmt = $db->prepare("INSERT OR REPLACE INTO login_attempts (ip_address, attempts, last_attempt)
                                      VALUES (?, ?, ?)");
                $stmt->execute([$ipAddress, $attempts, date('Y-m-d H:i:s')]);
            } else {
                // Se o login for bem-sucedido, limpar as tentativas
                $stmt = $db->prepare("DELETE FROM login_attempts WHERE ip_address = ?");
                $stmt->execute([$ipAddress]);
            }

            $db->commit();
        } catch (PDOException $e) {
            $db->rollBack();
            echo "Erro ao registrar a tentativa de login: " . $e->getMessage();
        }
    }

    // Verificação de login
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $username = trim($_POST['username']);
        $password = $_POST['password']; // Senha fornecida pelo usuário

        $ipAddress = $_SERVER['REMOTE_ADDR'];  // Obter o IP do usuário

        // Verificar as tentativas de login
        if (!checkLoginAttempts($ipAddress, $db)) {
            $error = "Você excedeu o número máximo de tentativas de login. Tente novamente mais tarde.";
        } else {
            // Consultar o banco de dados para encontrar o usuário
            $stmt = $db->prepare("SELECT * FROM users WHERE username = ?");
            $stmt->execute([$username]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            // Verificação com senha hash
            if ($user && password_verify($password, $user['password'])) {
                // Se o usuário for administrador, ignorar expiração
                if ($user['role'] === 'admin') {
                    $_SESSION['user'] = $user['username']; // Definindo a variável de sessão
                    $_SESSION['role'] = $user['role'];     // Definindo a role

                    // Registrar a tentativa de login bem-sucedida
                    logLoginAttempt($ipAddress, true, $db);

                    // Redirecionando para a página inicial após login
                    header("Location: index.php");
                    exit;
                } else {
                    // Verificar se o acesso do usuário expirou
                    $expires_at = $user['expires_at'];

                    // Se a data de expiração for menor que a data atual
                    if (strtotime($expires_at) < time()) {
                        $error = "Seu acesso expirou. Entre em contato com o suporte para reativação.";
                    } else {
                        $_SESSION['user'] = $user['username']; // Definindo a variável de sessão
                        $_SESSION['role'] = $user['role'];     // Definindo a role

                        // Registrar a tentativa de login bem-sucedida
                        logLoginAttempt($ipAddress, true, $db);

                        // Redirecionando para a página inicial após login
                        header("Location: index.php");
                        exit;
                    }
                }
            } else {
                // Login falhou, registrar tentativa
                logLoginAttempt($ipAddress, false, $db);
                $error = "Usuário ou senha inválidos.";
            }
        }
    }
} catch (PDOException $e) {
    echo "Erro ao conectar ao banco de dados: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <title>Login</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        /* Vídeo de fundo */
        body, html {
            height: 100%;
            margin: 0;
            padding: 0;
            position: relative;
        }

        video.background-video {
            position: fixed; /* Fixar o vídeo para que fique no fundo enquanto rola a página */
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            object-fit: cover;
            z-index: -1; /* Coloca o vídeo atrás do conteúdo */
        }

        .login-box {
            max-width: 400px;
            margin: 100px auto;
            background: rgba(0, 0, 0, 0.5); /* Fundo semitransparente para contraste */
            padding: 30px;
            border-radius: 10px;
            color: white;
            z-index: 1;
        }

        .form-control {
            background: #333;
            color: white;
            border: none;
        }

        .btn {
            font-size: 16px;
            padding: 12px 24px;
            border-radius: 30px;
            transition: background-color 0.3s ease, transform 0.2s ease;
        }

        .btn-primary {
            background-color: #ff7e5f;
            border: none;
        }

        .btn-primary:hover {
            background-color: #feb47b;
            transform: scale(1.05);
        }

        /* Responsividade */
        @media (max-width: 768px) {
            .login-box {
                max-width: 350px;
            }
        }
    </style>
</head>
<body>
    <!-- Vídeo Plexus como fundo -->
    <video class="background-video" autoplay loop muted>
        <source src="Plexus.mp4" type="video/mp4">
        Seu navegador não suporta vídeos.
    </video>

    <div class="login-box">
        <h2 class="text-center">🔐 Login</h2>
        <?php if ($error): ?><div class="alert alert-danger"><?= $error ?></div><?php endif; ?>
        <form method="POST">
            <input type="text" name="username" class="form-control mb-3" placeholder="Usuário" required>
            <input type="password" name="password" class="form-control mb-3" placeholder="Senha" required>
            <button type="submit" class="btn btn-primary w-100">Entrar</button>
        </form>
    </div>
</body>
</html>
